# API Examples

Complete examples for testing and integrating with the Video API.

## Postman/Insomnia Testing

### 1. Get All Videos

**Request:**
```
GET https://yourdomain.com/video-api/api/videos.php
```

**Headers:**
```
X-API-Key: your_api_key_here
```

**Expected Response:**
```json
{
  "success": true,
  "count": 15,
  "videos": [
    {
      "id": "vid_a1b2c3d4e5f6",
      "filename": "sample_video.mp4",
      "aspect_ratio": "16:9",
      "url": "https://yourdomain.com/video-api/api/stream.php?id=vid_a1b2c3d4e5f6",
      "file_size": "15728640",
      "duration": 45,
      "upload_date": "2025-11-25 14:30:00"
    }
  ]
}
```

### 2. Get Single Video

**Request:**
```
GET https://yourdomain.com/video-api/api/video.php?id=vid_a1b2c3d4e5f6
```

**Headers:**
```
X-API-Key: your_api_key_here
```

### 3. Stream Video

**Request:**
```
GET https://yourdomain.com/video-api/api/stream.php?id=vid_a1b2c3d4e5f6
```

**Headers:**
```
X-API-Key: your_api_key_here
```

## cURL Examples

### Get All Videos

```bash
curl -H "X-API-Key: your_api_key_here" \
     https://yourdomain.com/video-api/api/videos.php
```

### Get Single Video

```bash
curl -H "X-API-Key: your_api_key_here" \
     "https://yourdomain.com/video-api/api/video.php?id=vid_a1b2c3d4e5f6"
```

### Download Video

```bash
curl -H "X-API-Key: your_api_key_here" \
     "https://yourdomain.com/video-api/api/stream.php?id=vid_a1b2c3d4e5f6" \
     -o video.mp4
```

## Android Kotlin Examples

### 1. Setup Dependencies

**build.gradle (app):**
```gradle
dependencies {
    // Retrofit
    implementation 'com.squareup.retrofit2:retrofit:2.9.0'
    implementation 'com.squareup.retrofit2:converter-gson:2.9.0'

    // ExoPlayer (for video playback)
    implementation 'com.google.android.exoplayer:exoplayer:2.19.1'

    // Coroutines
    implementation 'org.jetbrains.kotlinx:kotlinx-coroutines-android:1.7.3'
}
```

### 2. Data Models

```kotlin
// Video.kt
data class Video(
    val id: String,
    val filename: String,
    val aspect_ratio: String,
    val url: String,
    val file_size: String,
    val duration: Int,
    val upload_date: String
)

// VideoResponse.kt
data class VideoResponse(
    val success: Boolean,
    val count: Int,
    val videos: List<Video>
)

// SingleVideoResponse.kt
data class SingleVideoResponse(
    val success: Boolean,
    val video: Video
)
```

### 3. API Service

```kotlin
// VideoApiService.kt
interface VideoApiService {
    @GET("api/videos.php")
    suspend fun getVideos(
        @Header("X-API-Key") apiKey: String
    ): VideoResponse

    @GET("api/video.php")
    suspend fun getVideo(
        @Query("id") videoId: String,
        @Header("X-API-Key") apiKey: String
    ): SingleVideoResponse
}

// ApiConfig.kt
object ApiConfig {
    const val BASE_URL = "https://yourdomain.com/video-api/"
    const val API_KEY = "your_api_key_here"

    fun getApiService(): VideoApiService {
        val retrofit = Retrofit.Builder()
            .baseUrl(BASE_URL)
            .addConverterFactory(GsonConverterFactory.create())
            .build()

        return retrofit.create(VideoApiService::class.java)
    }
}
```

### 4. Repository

```kotlin
// VideoRepository.kt
class VideoRepository {
    private val apiService = ApiConfig.getApiService()

    suspend fun getAllVideos(): Result<List<Video>> {
        return try {
            val response = apiService.getVideos(ApiConfig.API_KEY)
            if (response.success) {
                Result.success(response.videos)
            } else {
                Result.failure(Exception("Failed to fetch videos"))
            }
        } catch (e: Exception) {
            Result.failure(e)
        }
    }

    suspend fun getVideo(videoId: String): Result<Video> {
        return try {
            val response = apiService.getVideo(videoId, ApiConfig.API_KEY)
            if (response.success) {
                Result.success(response.video)
            } else {
                Result.failure(Exception("Failed to fetch video"))
            }
        } catch (e: Exception) {
            Result.failure(e)
        }
    }
}
```

### 5. ViewModel

```kotlin
// VideoViewModel.kt
class VideoViewModel : ViewModel() {
    private val repository = VideoRepository()

    private val _videos = MutableLiveData<List<Video>>()
    val videos: LiveData<List<Video>> = _videos

    private val _loading = MutableLiveData<Boolean>()
    val loading: LiveData<Boolean> = _loading

    private val _error = MutableLiveData<String?>()
    val error: LiveData<String?> = _error

    fun loadVideos() {
        viewModelScope.launch {
            _loading.value = true
            _error.value = null

            repository.getAllVideos()
                .onSuccess { videos ->
                    _videos.value = videos
                    _loading.value = false
                }
                .onFailure { exception ->
                    _error.value = exception.message
                    _loading.value = false
                }
        }
    }
}
```

### 6. RecyclerView Adapter

```kotlin
// VideoAdapter.kt
class VideoAdapter(
    private val onVideoClick: (Video) -> Unit
) : RecyclerView.Adapter<VideoAdapter.VideoViewHolder>() {

    private var videos = listOf<Video>()

    fun submitList(newVideos: List<Video>) {
        videos = newVideos
        notifyDataSetChanged()
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): VideoViewHolder {
        val binding = ItemVideoBinding.inflate(
            LayoutInflater.from(parent.context),
            parent,
            false
        )
        return VideoViewHolder(binding)
    }

    override fun onBindViewHolder(holder: VideoViewHolder, position: Int) {
        holder.bind(videos[position])
    }

    override fun getItemCount() = videos.size

    inner class VideoViewHolder(
        private val binding: ItemVideoBinding
    ) : RecyclerView.ViewHolder(binding.root) {

        fun bind(video: Video) {
            binding.apply {
                // Setup ExoPlayer
                val player = ExoPlayer.Builder(itemView.context).build()
                playerView.player = player

                // Prepare media item with API key header
                val mediaItem = MediaItem.Builder()
                    .setUri(video.url)
                    .setRequestMetadata(
                        MediaItem.RequestMetadata.Builder()
                            .setExtras(bundleOf("X-API-Key" to ApiConfig.API_KEY))
                            .build()
                    )
                    .build()

                player.setMediaItem(mediaItem)
                player.prepare()

                // Set aspect ratio
                playerView.resizeMode = when (video.aspect_ratio) {
                    "16:9" -> AspectRatioFrameLayout.RESIZE_MODE_FIT
                    "9:16" -> AspectRatioFrameLayout.RESIZE_MODE_ZOOM
                    else -> AspectRatioFrameLayout.RESIZE_MODE_FIT
                }

                root.setOnClickListener { onVideoClick(video) }
            }
        }
    }
}
```

### 7. Activity/Fragment

```kotlin
// MainActivity.kt
class MainActivity : AppCompatActivity() {

    private lateinit var binding: ActivityMainBinding
    private val viewModel: VideoViewModel by viewModels()
    private lateinit var adapter: VideoAdapter

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityMainBinding.inflate(layoutInflater)
        setContentView(binding.root)

        setupRecyclerView()
        observeViewModel()

        // Load videos
        viewModel.loadVideos()
    }

    private fun setupRecyclerView() {
        adapter = VideoAdapter { video ->
            // Handle video click
            Toast.makeText(this, "Playing: ${video.filename}", Toast.LENGTH_SHORT).show()
        }

        binding.recyclerView.apply {
            layoutManager = LinearLayoutManager(this@MainActivity)
            adapter = this@MainActivity.adapter
        }
    }

    private fun observeViewModel() {
        viewModel.videos.observe(this) { videos ->
            adapter.submitList(videos)
        }

        viewModel.loading.observe(this) { isLoading ->
            binding.progressBar.isVisible = isLoading
        }

        viewModel.error.observe(this) { error ->
            error?.let {
                Toast.makeText(this, it, Toast.LENGTH_LONG).show()
            }
        }
    }
}
```

### 8. Custom HTTP Client (for API key injection)

```kotlin
// OkHttpClient with Interceptor
object NetworkClient {
    fun getClient(): OkHttpClient {
        return OkHttpClient.Builder()
            .addInterceptor { chain ->
                val original = chain.request()
                val request = original.newBuilder()
                    .header("X-API-Key", ApiConfig.API_KEY)
                    .method(original.method, original.body)
                    .build()
                chain.proceed(request)
            }
            .connectTimeout(30, TimeUnit.SECONDS)
            .readTimeout(30, TimeUnit.SECONDS)
            .build()
    }
}

// Updated Retrofit builder
fun getApiService(): VideoApiService {
    val retrofit = Retrofit.Builder()
        .baseUrl(ApiConfig.BASE_URL)
        .client(NetworkClient.getClient())
        .addConverterFactory(GsonConverterFactory.create())
        .build()

    return retrofit.create(VideoApiService::class.java)
}
```

## JavaScript/Web Examples

### Fetch API

```javascript
// Get all videos
async function getAllVideos() {
  try {
    const response = await fetch('https://yourdomain.com/video-api/api/videos.php', {
      headers: {
        'X-API-Key': 'your_api_key_here'
      }
    });

    const data = await response.json();

    if (data.success) {
      console.log('Videos:', data.videos);
      return data.videos;
    }
  } catch (error) {
    console.error('Error:', error);
  }
}

// Play video
function playVideo(videoId) {
  const videoElement = document.getElementById('player');
  videoElement.src = `https://yourdomain.com/video-api/api/stream.php?id=${videoId}`;

  // Add API key to request (using fetch with blob)
  fetch(videoElement.src, {
    headers: {
      'X-API-Key': 'your_api_key_here'
    }
  })
  .then(response => response.blob())
  .then(blob => {
    videoElement.src = URL.createObjectURL(blob);
    videoElement.play();
  });
}
```

### Axios

```javascript
import axios from 'axios';

const api = axios.create({
  baseURL: 'https://yourdomain.com/video-api/',
  headers: {
    'X-API-Key': 'your_api_key_here'
  }
});

// Get all videos
async function getVideos() {
  try {
    const response = await api.get('api/videos.php');
    return response.data.videos;
  } catch (error) {
    console.error('Error:', error);
  }
}
```

## Error Handling

### Common Error Responses

**401 Unauthorized:**
```json
{
  "success": false,
  "error": "Invalid API key"
}
```

**404 Not Found:**
```json
{
  "success": false,
  "error": "Video not found"
}
```

**500 Internal Server Error:**
```json
{
  "success": false,
  "error": "Internal server error"
}
```

### Handle Errors in Android

```kotlin
try {
    val response = apiService.getVideos(ApiConfig.API_KEY)
    if (response.success) {
        // Success
    }
} catch (e: HttpException) {
    when (e.code()) {
        401 -> Log.e("API", "Invalid API key")
        404 -> Log.e("API", "Not found")
        500 -> Log.e("API", "Server error")
        else -> Log.e("API", "Error: ${e.message}")
    }
} catch (e: IOException) {
    Log.e("API", "Network error: ${e.message}")
}
```

## Testing Checklist

- [ ] API returns 200 OK with valid API key
- [ ] API returns 401 with invalid API key
- [ ] Videos array is properly formatted
- [ ] Video streaming works
- [ ] Range requests work (seeking)
- [ ] CORS headers allow app access
- [ ] All video metadata is correct

---

Happy coding! 🚀
