# Deployment Guide

Complete step-by-step guide to deploy the Video API Backend to your cPanel/Namecheap VPS.

## Pre-Deployment Checklist

### 1. Generate API Key

Generate a secure 32-character API key:

**Option 1: Online Generator**
- Visit: https://randomkeygen.com/
- Use the "Fort Knox Password" section

**Option 2: Command Line**
```bash
# On Mac/Linux
openssl rand -hex 16

# On Windows (PowerShell)
-join ((48..57) + (65..90) + (97..122) | Get-Random -Count 32 | ForEach-Object {[char]$_})
```

### 2. Generate Admin Password Hash

Generate a bcrypt hash for your admin password:

```bash
php -r "echo password_hash('your_secure_password', PASSWORD_BCRYPT);"
```

Or use an online tool: https://bcrypt-generator.com/

## Deployment Steps

### Step 1: Update Configuration

Edit `api/config.php`:

```php
// Replace with your generated API key
define('API_KEY', 'your_32_character_api_key_here');

// Replace with your generated password hash
define('ADMIN_PASSWORD_HASH', '$2y$10$your_hash_here');

// Update with your actual domain
define('BASE_URL', 'https://yourdomain.com/video-api');
```

### Step 2: Upload to Server

**Using cPanel File Manager:**

1. Login to cPanel
2. Open File Manager
3. Navigate to `/public_html/`
4. Create folder `video-api`
5. Upload all files to `/public_html/video-api/`

**Using FTP/SFTP:**

```bash
# Upload entire directory
ftp yourdomain.com
cd public_html
mkdir video-api
cd video-api
put -r *
```

### Step 3: Set Permissions

In cPanel File Manager or via SSH:

```bash
chmod 755 videos/
chmod 755 database/
chmod 755 admin/assets/
chmod 644 api/config.php
chmod 644 .htaccess
```

### Step 4: Verify PHP Settings

Check PHP version and settings in cPanel:

**Required:**
- PHP 7.4 or higher
- PDO extension (for SQLite)
- JSON extension
- FileInfo extension

**Recommended settings:**
```ini
upload_max_filesize = 100M
post_max_size = 100M
max_execution_time = 300
memory_limit = 256M
```

### Step 5: Test Installation

1. **Test Root Page**
   - Visit: `https://yourdomain.com/video-api/`
   - Should see API documentation page

2. **Test Admin Login**
   - Visit: `https://yourdomain.com/video-api/admin/login.php`
   - Login with username: `admin` and your password
   - Should redirect to dashboard

3. **Test Upload**
   - Upload a test video
   - Check if it appears in the video list

4. **Test API (using Postman or curl)**

   ```bash
   # Test videos endpoint
   curl -H "X-API-Key: your_api_key_here" \
        https://yourdomain.com/video-api/api/videos.php

   # Should return JSON with empty videos array
   ```

### Step 6: Enable HTTPS

**In cPanel:**

1. Go to "SSL/TLS Status"
2. Enable SSL for your domain
3. Update `api/config.php` to use `https://`

**Force HTTPS (add to .htaccess):**

```apache
RewriteEngine On
RewriteCond %{HTTPS} off
RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
```

## Post-Deployment

### Security Hardening

1. **Change Default Password**
   - Generate new password hash
   - Update `api/config.php`

2. **Rotate API Key**
   - Generate new API key
   - Update `api/config.php`
   - Update Android app with new key

3. **Restrict Admin Access (Optional)**

   Add to `admin/.htaccess`:
   ```apache
   Order Deny,Allow
   Deny from all
   Allow from your.ip.address
   ```

4. **Enable Error Logging**

   Create `logs/` directory:
   ```bash
   mkdir logs
   chmod 755 logs
   ```

### Backup Strategy

Create a backup script (`backup.sh`):

```bash
#!/bin/bash
DATE=$(date +%Y%m%d)
BACKUP_DIR="backups"

# Create backup directory
mkdir -p $BACKUP_DIR

# Backup database
cp database/videos.db $BACKUP_DIR/videos_$DATE.db

# Backup videos
tar -czf $BACKUP_DIR/videos_$DATE.tar.gz videos/

# Keep only last 7 days
find $BACKUP_DIR -name "*.db" -mtime +7 -delete
find $BACKUP_DIR -name "*.tar.gz" -mtime +7 -delete
```

Set up cron job in cPanel:
```
0 3 * * * /path/to/backup.sh
```

## Android App Integration

### Update API Configuration

In your Android app, update:

```kotlin
object ApiConfig {
    const val BASE_URL = "https://yourdomain.com/video-api/"
    const val API_KEY = "your_api_key_here"
}
```

### Retrofit Setup

```kotlin
val retrofit = Retrofit.Builder()
    .baseUrl(ApiConfig.BASE_URL)
    .addConverterFactory(GsonConverterFactory.create())
    .build()

val apiService = retrofit.create(VideoApiService::class.java)
```

### Test API from Android

```kotlin
// Get all videos
apiService.getVideos(ApiConfig.API_KEY).enqueue(object : Callback<VideoResponse> {
    override fun onResponse(call: Call<VideoResponse>, response: Response<VideoResponse>) {
        if (response.isSuccessful) {
            val videos = response.body()?.videos
            Log.d("API", "Videos: ${videos?.size}")
        }
    }

    override fun onFailure(call: Call<VideoResponse>, t: Throwable) {
        Log.e("API", "Error: ${t.message}")
    }
})
```

## Troubleshooting

### 500 Internal Server Error

**Check:**
1. PHP error logs in cPanel
2. File permissions (755 for directories, 644 for files)
3. `.htaccess` syntax errors

**Solution:**
```bash
# Check PHP error log
tail -f ~/public_html/video-api/error_log
```

### Database Not Found

**Cause:** Database file not created yet

**Solution:** Visit any page once to auto-initialize database

### Upload Fails

**Check:**
1. PHP upload limits in cPanel
2. Directory permissions (videos/ should be 755)
3. Disk space available

**Increase limits in cPanel:**
- MultiPHP INI Editor → upload_max_filesize = 100M

### API Returns 401 Unauthorized

**Check:**
1. API key matches in config and Android app
2. Header name is exactly `X-API-Key`
3. No extra spaces in API key

### Videos Won't Play

**Check:**
1. Video format (MP4 H.264 recommended)
2. File permissions (644)
3. CORS headers in .htaccess

## Monitoring

### Check Logs

```bash
# Application logs
tail -f logs/app.log

# PHP error logs
tail -f ~/public_html/video-api/error_log

# Apache access logs
tail -f ~/access-logs/yourdomain.com
```

### Monitor Disk Space

```bash
# Check disk usage
du -sh videos/
du -sh database/

# Check total space
df -h
```

## Maintenance

### Clean Up Old Videos

Periodically check and remove unused videos through admin panel.

### Update API

To update code:
1. Backup current installation
2. Upload new files
3. Keep `api/config.php` unchanged
4. Test thoroughly

## Support Checklist

- [ ] API key generated and configured
- [ ] Admin password changed from default
- [ ] HTTPS enabled
- [ ] Test video uploaded successfully
- [ ] API tested with Postman/curl
- [ ] Android app can fetch videos
- [ ] Backup strategy implemented
- [ ] Error logging enabled
- [ ] Disk space monitored

## Contact

For deployment issues:
- Check server error logs first
- Verify PHP version and extensions
- Test with simple curl commands
- Contact hosting support if needed

---

**Deployment Complete!** 🚀

Your Video API Backend is now live and ready for use.
