// Video API Admin Panel JavaScript

document.addEventListener('DOMContentLoaded', function() {
    const uploadForm = document.getElementById('uploadForm');
    const uploadArea = document.getElementById('uploadArea');
    const videoFile = document.getElementById('videoFile');
    const progressContainer = document.getElementById('progressContainer');
    const progressFill = document.getElementById('progressFill');
    const progressText = document.getElementById('progressText');
    const uploadAlert = document.getElementById('uploadAlert');
    const uploadBtn = document.getElementById('uploadBtn');

    if (uploadArea && videoFile) {
        // Click to select file
        uploadArea.addEventListener('click', function() {
            videoFile.click();
        });

        // File selected
        videoFile.addEventListener('change', function() {
            if (this.files.length > 0) {
                updateUploadArea(this.files[0]);
            }
        });

        // Drag and drop
        uploadArea.addEventListener('dragover', function(e) {
            e.preventDefault();
            this.classList.add('drag-over');
        });

        uploadArea.addEventListener('dragleave', function() {
            this.classList.remove('drag-over');
        });

        uploadArea.addEventListener('drop', function(e) {
            e.preventDefault();
            this.classList.remove('drag-over');

            const files = e.dataTransfer.files;
            if (files.length > 0) {
                videoFile.files = files;
                updateUploadArea(files[0]);
            }
        });
    }

    // Handle form submission
    if (uploadForm) {
        uploadForm.addEventListener('submit', async function(e) {
            e.preventDefault();

            if (!videoFile.files || videoFile.files.length === 0) {
                showAlert('Please select a video file', 'error');
                return;
            }

            const formData = new FormData(this);

            try {
                uploadBtn.disabled = true;
                progressContainer.style.display = 'block';
                uploadAlert.style.display = 'none';

                const response = await uploadVideo(formData);

                if (response.success) {
                    showAlert('Video uploaded successfully!', 'success');
                    progressFill.style.width = '100%';
                    progressText.textContent = 'Upload complete!';

                    // Reload page after 2 seconds
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    throw new Error(response.error || 'Upload failed');
                }
            } catch (error) {
                showAlert(error.message, 'error');
                progressContainer.style.display = 'none';
                uploadBtn.disabled = false;
            }
        });
    }

    // Update upload area text
    function updateUploadArea(file) {
        const content = uploadArea.querySelector('.upload-content p');
        if (content) {
            content.textContent = `Selected: ${file.name}`;
        }
    }

    // Upload video with progress
    async function uploadVideo(formData) {
        return new Promise((resolve, reject) => {
            const xhr = new XMLHttpRequest();

            // Upload progress
            xhr.upload.addEventListener('progress', function(e) {
                if (e.lengthComputable) {
                    const percentComplete = (e.loaded / e.total) * 100;
                    progressFill.style.width = percentComplete + '%';
                    progressText.textContent = `Uploading: ${Math.round(percentComplete)}%`;
                }
            });

            // Upload complete
            xhr.addEventListener('load', function() {
                if (xhr.status === 200) {
                    try {
                        const response = JSON.parse(xhr.responseText);
                        resolve(response);
                    } catch (e) {
                        reject(new Error('Invalid server response'));
                    }
                } else {
                    try {
                        const response = JSON.parse(xhr.responseText);
                        reject(new Error(response.error || 'Upload failed'));
                    } catch (e) {
                        reject(new Error('Upload failed'));
                    }
                }
            });

            // Upload error
            xhr.addEventListener('error', function() {
                reject(new Error('Network error'));
            });

            xhr.open('POST', 'upload_handler.php');
            xhr.send(formData);
        });
    }

    // Show alert message
    function showAlert(message, type) {
        uploadAlert.textContent = message;
        uploadAlert.className = 'alert alert-' + type;
        uploadAlert.style.display = 'block';

        // Auto-hide success messages
        if (type === 'success') {
            setTimeout(() => {
                uploadAlert.style.display = 'none';
            }, 5000);
        }
    }
});

// Delete video function
async function deleteVideo(videoId) {
    if (!confirm('Are you sure you want to delete this video? This action cannot be undone.')) {
        return;
    }

    try {
        const response = await fetch('delete_handler.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ video_id: videoId })
        });

        const data = await response.json();

        if (data.success) {
            // Remove video card from DOM
            const videoCard = document.querySelector(`[data-video-id="${videoId}"]`);
            if (videoCard) {
                videoCard.style.transition = 'all 0.3s';
                videoCard.style.opacity = '0';
                videoCard.style.transform = 'scale(0.8)';

                setTimeout(() => {
                    videoCard.remove();

                    // Check if no videos left
                    const videosGrid = document.querySelector('.videos-grid');
                    if (videosGrid && videosGrid.children.length === 0) {
                        window.location.reload();
                    }
                }, 300);
            }

            alert('Video deleted successfully!');

            // Update video counter
            window.location.reload();
        } else {
            throw new Error(data.error || 'Delete failed');
        }
    } catch (error) {
        alert('Failed to delete video: ' + error.message);
    }
}
