<?php
/**
 * GET /api/stream.php?id={video_id}
 * Streams video file with API key validation
 * Supports range requests for seeking
 * Requires: X-API-Key header
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/helpers.php';

// Validate API key
requireAPIKey();

// Get video ID from query parameter
$videoId = $_GET['id'] ?? null;

if (!$videoId) {
    http_response_code(400);
    die('Missing video ID parameter');
}

try {
    // Get video from database
    $video = getVideoById($videoId);

    if (!$video) {
        http_response_code(404);
        die('Video not found');
    }

    // Get video file path
    $filePath = UPLOAD_DIR . $video['filename'];

    if (!file_exists($filePath)) {
        error_log("Video file not found: $filePath");
        http_response_code(404);
        die('Video file not found');
    }

    // Get file info
    $fileSize = filesize($filePath);
    $mimeType = mime_content_type($filePath);

    // Handle range requests (for seeking)
    $start = 0;
    $end = $fileSize - 1;
    $length = $fileSize;

    if (isset($_SERVER['HTTP_RANGE'])) {
        // Parse range header
        if (preg_match('/bytes=(\d+)-(\d*)/', $_SERVER['HTTP_RANGE'], $matches)) {
            $start = intval($matches[1]);
            if (!empty($matches[2])) {
                $end = intval($matches[2]);
            }
            $length = $end - $start + 1;

            // Send partial content status
            http_response_code(206);
            header("Content-Range: bytes $start-$end/$fileSize");
        }
    }

    // Set headers
    header("Content-Type: $mimeType");
    header("Content-Length: $length");
    header("Accept-Ranges: bytes");
    header("Cache-Control: public, max-age=3600");
    header("Content-Disposition: inline; filename=\"" . basename($video['original_filename']) . "\"");

    // Disable output buffering
    if (ob_get_level()) {
        ob_end_clean();
    }

    // Open file and seek to start position
    $file = fopen($filePath, 'rb');
    if ($start > 0) {
        fseek($file, $start);
    }

    // Stream the file in chunks
    $chunkSize = 8192; // 8KB chunks
    $bytesRemaining = $length;

    while ($bytesRemaining > 0 && !feof($file)) {
        $readSize = min($chunkSize, $bytesRemaining);
        $buffer = fread($file, $readSize);

        if ($buffer === false) {
            break;
        }

        echo $buffer;
        flush();

        $bytesRemaining -= strlen($buffer);
    }

    fclose($file);
    exit;

} catch (Exception $e) {
    error_log("Stream error: " . $e->getMessage());
    http_response_code(500);
    die('Internal server error');
}
?>
