<?php
// Admin Authentication (Session-based)

require_once __DIR__ . '/../api/config.php';

/**
 * Check if admin is logged in
 * @return bool True if logged in, false otherwise
 */
function isAdminLoggedIn() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
        return false;
    }

    // Check session timeout
    if (isset($_SESSION['last_activity'])) {
        $elapsedTime = time() - $_SESSION['last_activity'];
        if ($elapsedTime > SESSION_TIMEOUT) {
            destroyAdminSession();
            return false;
        }
    }

    // Update last activity time
    $_SESSION['last_activity'] = time();

    return true;
}

/**
 * Require admin authentication (redirect to login if not authenticated)
 */
function requireAdminAuth() {
    if (!isAdminLoggedIn()) {
        header('Location: login.php');
        exit;
    }
}

/**
 * Verify admin credentials
 * @param string $username Username
 * @param string $password Password
 * @return bool True if valid, false otherwise
 */
function verifyAdminCredentials($username, $password) {
    if ($username !== ADMIN_USERNAME) {
        return false;
    }

    return password_verify($password, ADMIN_PASSWORD_HASH);
}

/**
 * Login admin user
 * @param string $username Username
 * @param string $password Password
 * @return bool Success status
 */
function loginAdmin($username, $password) {
    if (!verifyAdminCredentials($username, $password)) {
        return false;
    }

    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    // Regenerate session ID to prevent session fixation
    session_regenerate_id(true);

    $_SESSION['admin_logged_in'] = true;
    $_SESSION['admin_username'] = $username;
    $_SESSION['login_time'] = time();
    $_SESSION['last_activity'] = time();

    return true;
}

/**
 * Logout admin user
 */
function logoutAdmin() {
    destroyAdminSession();
    header('Location: login.php');
    exit;
}

/**
 * Destroy admin session
 */
function destroyAdminSession() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    $_SESSION = [];

    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }

    session_destroy();
}

/**
 * Get session remaining time in seconds
 * @return int Seconds remaining
 */
function getSessionRemainingTime() {
    if (!isAdminLoggedIn()) {
        return 0;
    }

    $elapsed = time() - $_SESSION['last_activity'];
    $remaining = SESSION_TIMEOUT - $elapsed;

    return max(0, $remaining);
}
?>
